#
# Gramps - a GTK+/GNOME based genealogy program
#
# Copyright (C) 2010  Michiel D. Nauta
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, see <https://www.gnu.org/licenses/>.
#

"""
Provide merge capabilities for repositories.
"""

# -------------------------------------------------------------------------
#
# Gramps modules
#
# -------------------------------------------------------------------------
from ..lib import Source, Note
from ..db import DbTxn
from ..const import GRAMPS_LOCALE as glocale

_ = glocale.translation.sgettext
from ..errors import MergeError


# -------------------------------------------------------------------------
#
# MergeRepoQuery
#
# -------------------------------------------------------------------------
class MergeRepositoryQuery:
    """
    Create database query to merge two repositories.
    """

    def __init__(self, dbstate, phoenix, titanic):
        self.database = dbstate.db
        self.phoenix = phoenix
        self.titanic = titanic

    def execute(self):
        """
        Merges two repositories into a single repository.
        """
        new_handle = self.phoenix.get_handle()
        old_handle = self.titanic.get_handle()

        self.phoenix.merge(self.titanic)

        with DbTxn(_("Merge Repositories"), self.database) as trans:
            self.database.commit_repository(self.phoenix, trans)
            for class_name, handle in self.database.find_backlink_handles(old_handle):
                if class_name == Source.__name__:
                    source = self.database.get_source_from_handle(handle)
                    assert source.has_handle_reference("Repository", old_handle)
                    source.replace_repo_references(old_handle, new_handle)
                    self.database.commit_source(source, trans)
                elif class_name == Note.__name__:
                    note = self.database.get_note_from_handle(handle)
                    assert note.has_handle_reference("Repository", old_handle)
                    note.replace_handle_reference("Repository", old_handle, new_handle)
                    self.database.commit_note(note, trans)
                else:
                    raise MergeError(
                        "Encounter an object of type %s that has "
                        "a repository reference." % class_name
                    )
            self.database.remove_repository(old_handle, trans)
